/*
** WebCamApp.c
**
** Copyright  2010 Future Technology Devices International Limited
**
**  C Source file for Vinculum II sample application
** Main module
**
** Author: FTDI
** Project: Vinculum II
** Module: Vinculum II Sample Applications
** Requires: VOS UART USBHost GPIO //UVC
** Comments:
**
** History:
**  1  Initial version
**
*/
#include "vos.h"
#include "USBHost.h"
#include "USB.h"
#include "GPIO.h"
#include "FIFO.h"
#include "UVC.h"
#include "SEPS525.h"


#define SIZEOF_FIRMWARE_TASK_MEMORY 0x800
#define VOS_QUANTUM                 50
#define VOS_TICK_INTERVAL           1


#define NUMBER_OF_DEVICES           4
#define VOS_DEV_USBHOST             0
#define VOS_DEV_FIFO                1
#define VOS_DEV_GPIO                2
#define VOS_DEV_UVC                 3


VOS_HANDLE hUsbHost, hFifo, hGpio;
VOS_HANDLE hUvc;

vos_tcb_t *tcbUsbRd, *tcbDispOut;


// GPIO context structure
gpio_context_t gpioContext;
usbhost_context_t usbhostContext;

fifo_context_t fifo_ctx;

void usbReader(void);
void displayOutput(void);


#define ISO_TRANSFER_SIZE     192
#define ISO_TRANSFER_COUNT    1
#define WEBCAM_HEADER_SIZE    12
#define WEBCAM_PAYLOAD_SIZE   (ISO_TRANSFER_SIZE - WEBCAM_HEADER_SIZE)


#define BUF_SIZE              (ISO_TRANSFER_SIZE * ISO_TRANSFER_COUNT)

// buffers and mutexes to protect them
unsigned char buffer1[BUF_SIZE];
unsigned char buffer2[BUF_SIZE];
vos_semaphore_t sb1_avail, sb1_full;
vos_semaphore_t sb2_avail, sb2_full;


#ifdef _VNC2
port vII_gpio_data_tx_pa_1@0x186;
port vII_gpio_data_tx_pb_1@0x187;
port vII_gpio_data_tx_pc_1@0x188;
#endif

unsigned int data;
unsigned int pxcount;
unsigned int xfercount;


typedef enum _VS_InterfaceControlStatus_e
{
    VS_CONTROL_UNDEFINED=0x00,        //0x00
    VS_PROBE_CONTROL,                 //0x01
    VS_COMMIT_CONTROL,                //0x02
    VS_STILL_PROBE_CONTROL,           //0x03
    VS_STILL_COMMIT_CONTROL,          //0x04
    VS_STILL_IMAGE_TRIGGER_CONTROL,   //0x05
    VS_STREAM_ERROR_CODE_CONTROL,     //0x06
    VS_GENERATE_KEY_FRAME_CONTROL,    //0x07
    VS_UPDATE_FRAME_SEGMENT_CONTROL,  //0x08
    VS_SYNCH_DELAY_CONTROL            //0x09
}VS_InterfaceControlStatus_e;

typedef enum _VS_ControlRequest_e
{
    RC_UNDEFINED=0x00,   //0x00
    SET_CUR,             //0x01
    GET_CUR= 0x81,       //0x81
    GET_MIN,             //0x82
    GET_MAX,             //0x83
    GET_RES,             //0x84
    GET_LEN,             //0x85
    GET_INFO,            //0x86
    GET_DEF              //0x87
}VS_ControlRequest_e;

typedef struct _VideoProbeAndCommiteControl_t
{

    unsigned short bmHint;
    unsigned char bFormatIndex;
    unsigned char bFrameIndex;
    unsigned int dwFrameInterval;
    unsigned short wKeyFrameRate;
    unsigned short wPFrameRate;
    unsigned short wCompQuality;
    unsigned short wCompWindowSize;
    unsigned short wDelay;
    unsigned int dwMaxVideoFrameSize;
    unsigned int dwMaxPayloadTransferSize;
    unsigned int dwClockFrequency;
    unsigned char bmFramingInfo;
    unsigned char bPreferedVersion;
    unsigned char bMinVersion;
    unsigned char bMaxVersion;

} VideoProbeAndCommiteControl_t;

VideoProbeAndCommiteControl_t VProbeAndCom;
VideoProbeAndCommiteControl_t VProbeAndComMax;
VideoProbeAndCommiteControl_t VProbeAndComMin;


//void yuv2rgb(void);
void yuv2rgb(unsigned int addr,unsigned char short_packet);
void process_data_stream(unsigned char *buf,unsigned short num_bytes);


void main(void)
{
    // GPIO IOCTL request block
    gpio_ioctl_cb_t gpio_iocb;
    unsigned char packageType;


    vos_set_clock_frequency(VOS_48MHZ_CLOCK_FREQUENCY);
    vos_init(VOS_QUANTUM, VOS_TICK_INTERVAL, NUMBER_OF_DEVICES);


    packageType = vos_get_package_type();

    if (packageType == VINCULUM_II_32_PIN) {
        asm{HALT};  // 32-pin package not supported for the current board setup
    }
    else if (packageType == VINCULUM_II_48_PIN){
        asm{HALT};  // 48-pin package not supported for the current board setup
    }
    else if (packageType == VINCULUM_II_64_PIN) {
    // DATA INTERFACE
        // GPIO Port A 0 to pin 28 as Bi-Directional.
        vos_iomux_define_bidi(19, IOMUX_IN_GPIO_PORT_A_0, IOMUX_OUT_GPIO_PORT_A_0);   // IOBUS8
        // GPIO Port A 1 to pin 29 as Bi-Directional.
        vos_iomux_define_bidi(20, IOMUX_IN_GPIO_PORT_A_1, IOMUX_OUT_GPIO_PORT_A_1);   // IOBUS9
        // GPIO Port A 2 to pin 31 as Bi-Directional.
        vos_iomux_define_bidi(22, IOMUX_IN_GPIO_PORT_A_2, IOMUX_OUT_GPIO_PORT_A_2);   // IOBUS10
        // GPIO Port A 3 to pin 32 as Bi-Directional.
        vos_iomux_define_bidi(23, IOMUX_IN_GPIO_PORT_A_3, IOMUX_OUT_GPIO_PORT_A_3);   // IOBUS11
        // GPIO Port A 4 to pin 39 as Bi-Directional.
        vos_iomux_define_bidi(24, IOMUX_IN_GPIO_PORT_A_4, IOMUX_OUT_GPIO_PORT_A_4);   // IOBUS12
        // GPIO Port A 5 to pin 40 as Bi-Directional.
        vos_iomux_define_bidi(25, IOMUX_IN_GPIO_PORT_A_5, IOMUX_OUT_GPIO_PORT_A_5);   // IOBUS13
        // GPIO Port A 6 to pin 41 as Bi-Directional.
        vos_iomux_define_bidi(26, IOMUX_IN_GPIO_PORT_A_6, IOMUX_OUT_GPIO_PORT_A_6);   // IOBUS14
        // GPIO Port A 7 to pin 42 as Bi-Directional.
        vos_iomux_define_bidi(27, IOMUX_IN_GPIO_PORT_A_7, IOMUX_OUT_GPIO_PORT_A_7);   // IOBUS15
    // CONTROL INTERFACE
        // GPIO Port B 0 to pin 43 as Output. - WR
        vos_iomux_define_output(28, IOMUX_OUT_GPIO_PORT_B_0);   // IOBUS16
        // GPIO Port B 1 to pin 44 as Output. - RD
        vos_iomux_define_output(29, IOMUX_OUT_GPIO_PORT_B_1);   // IOBUS17
        // GPIO Port B 2 to pin 45 as Output. - CS
        vos_iomux_define_output(31, IOMUX_OUT_GPIO_PORT_B_2);   // IOBUS18
        // GPIO Port B 3 to pin 46 as Output. - RS
        vos_iomux_define_output(32, IOMUX_OUT_GPIO_PORT_B_3);   // IOBUS19
        // GPIO Port B 4 to pin 47 as Output. - DOTCLK
        vos_iomux_define_output(39, IOMUX_OUT_GPIO_PORT_B_4);   // IOBUS20
//        vos_iomux_define_output(57, IOMUX_OUT_GPIO_PORT_B_4);
        // GPIO Port B 5 to pin 48 as Output. - HSYNC
        vos_iomux_define_output(48, IOMUX_OUT_GPIO_PORT_B_5);   // IOBUS29
        // GPIO Port B 6 to pin 49 as Output. - VSYNC
        vos_iomux_define_output(49, IOMUX_OUT_GPIO_PORT_B_6);   // IOBUS30
        // GPIO Port B 7 to pin 50 as Output. - ENABLE
        vos_iomux_define_output(50, IOMUX_OUT_GPIO_PORT_B_7);   // IOBUS31
        // GPIO Port D 6 to pin 55 as Output. - RES
        vos_iomux_define_output(55, IOMUX_OUT_GPIO_PORT_D_6);   // IOBUS34
        // GPIO Port D 7 to pin 56 as Input.
        vos_iomux_define_input(56, IOMUX_IN_GPIO_PORT_D_7);   // IOBUS35
    }

    gpioContext.port_identifier = GPIO_PORT_A;
    gpio_init(VOS_DEV_GPIO, &gpioContext);

    // configure USB Host port 1 only
     // use a max of 4 USB devices
    usbhostContext.if_count = 16;
    usbhostContext.ep_count = 10;
    usbhostContext.xfer_count = 2;
    usbhostContext.iso_xfer_count = 36;

    usbhost_init(VOS_DEV_USBHOST, -1, &usbhostContext);

    uvc_init(VOS_DEV_UVC);

    // do any required setup on devices
    tcbUsbRd = vos_create_thread(31,SIZEOF_FIRMWARE_TASK_MEMORY,usbReader,0);
    if (tcbUsbRd == 0xffff)
    {
        asm{HALT};
    }

    tcbDispOut = vos_create_thread(30,SIZEOF_FIRMWARE_TASK_MEMORY,displayOutput,0);

    if (tcbDispOut == 0xffff)
    {
        asm{HALT};
    }

    vos_init_semaphore(&sb1_avail, 1);
    vos_init_semaphore(&sb1_full, 0);

    vos_init_semaphore(&sb2_avail, 1);
    vos_init_semaphore(&sb2_full, 0);

//    SEPS525_init();

    vos_start_scheduler();

main_loop:
    goto main_loop;
}




void usbReader(void)
{
   // USBHost ioctl request block
    usbhost_ioctl_cb_t hc_iocb;

    usbhost_ioctl_cb_class_t hc_iocb_class;
    usbhost_ioctl_cb_class_t hc_class;
   // endpoint handles
    usbhost_ep_handle *epIsoIn, *epIsoOut, *epCtrl;
    // endpoint info
    usbhost_ioctl_cb_ep_info_t epInfo;
    // device request
    usb_deviceRequest_t desc_dev;
    usbhost_xfer_iso_t xfer;

    unsigned short frame;
    usbhost_ioctl_cb_dev_info_t devInfo;
    int ifs, aset;

    vos_semaphore_t semDum;
    unsigned char num_dev;
    unsigned char status, ret;


    // device handle
    usbhost_device_handle *ifDev;
    unsigned char i;
    char *buf, c;
    static int ind = 0;

    hGpio = vos_dev_open(VOS_DEV_GPIO);
    hUsbHost = vos_dev_open(VOS_DEV_USBHOST);
    hUvc = vos_dev_open(VOS_DEV_UVC);
    do
    {
        uvc_ioctl_cb_t uvc_iocb;

        uvc_iocb.ioctl_code = VOS_IOCTL_UVC_ATTACH;
        uvc_iocb.set = (void *) hUsbHost;

        if (vos_dev_ioctl(hUvc, &uvc_iocb) != UVC_OK)
        {
            asm{HALT};
        }

        //GET_CUR = 0x81
        set_class_request(desc_dev,1,0,1,GET_CUR,26);

        uvc_iocb.ioctl_code = VOS_IOCTL_UVC_CLASS_REQUEST;
        uvc_iocb.set = &desc_dev;
        uvc_iocb.get = &VProbeAndCom;

        if (vos_dev_ioctl(hUvc, &uvc_iocb) != UVC_OK)
        {
            asm{HALT};
        }

        //GET_MAX = 0x83
        set_class_request(desc_dev,1,0,1,GET_MAX,26);

        uvc_iocb.ioctl_code = VOS_IOCTL_UVC_CLASS_REQUEST;
        uvc_iocb.set = &desc_dev;
        uvc_iocb.get = &VProbeAndComMax;

        if (vos_dev_ioctl(hUvc, &uvc_iocb) != UVC_OK)
        {
            asm{HALT};
        }

        //GET_MIN = 0x82
        set_class_request(desc_dev,1,0,1,GET_MIN,26);

        uvc_iocb.ioctl_code = VOS_IOCTL_UVC_CLASS_REQUEST;
        uvc_iocb.set = &desc_dev;
        uvc_iocb.get = &VProbeAndComMin;

        if (vos_dev_ioctl(hUvc, &uvc_iocb) != UVC_OK)
        {
            asm{HALT};
        }

        //SET_CUR Change the interval to 2000000 i.e 5 frames per second
        //change format descriptor to 1 Uncompressed  Video Format
        //By default the Frame Index is 2(i.e)160*120
        //Probe to make sure ISO rate is ISO_TRANSFER_SIZE

        // maximum frame rate
        VProbeAndCom.dwFrameInterval = 2000000;
        VProbeAndCom.bFormatIndex = 1;
        VProbeAndCom.bFrameIndex = 2;
        // request data in the format that the Linux app expects it
        VProbeAndCom.dwMaxVideoFrameSize = 38400;
        VProbeAndCom.dwMaxPayloadTransferSize = ISO_TRANSFER_SIZE;

        set_class_request(desc_dev,1,0,1,SET_CUR,26);

        uvc_iocb.ioctl_code = VOS_IOCTL_UVC_CLASS_REQUEST;
        uvc_iocb.set = &desc_dev;
        uvc_iocb.get = &VProbeAndCom;

        if (vos_dev_ioctl(hUvc, &uvc_iocb) != UVC_OK)
        {
            asm{HALT};
        }

        //GET_CUR
        set_class_request(desc_dev,1,0,1,GET_CUR,26);

        uvc_iocb.ioctl_code = VOS_IOCTL_UVC_CLASS_REQUEST;
        uvc_iocb.set = &desc_dev;
        uvc_iocb.get = &VProbeAndCom;

        if (vos_dev_ioctl(hUvc, &uvc_iocb) != UVC_OK)
        {
            asm{HALT};
        }

        // at this stage can check returned values to ensure they are set
        // however the dwMaxPayloadTransferSize depends on the endpoint
        // we are using so may differ from that expected

        //SET_CUR Commite
        set_class_request(desc_dev,1,0,2,SET_CUR,26);

        uvc_iocb.ioctl_code = VOS_IOCTL_UVC_CLASS_REQUEST;
        uvc_iocb.set = &desc_dev;
        uvc_iocb.get = &VProbeAndCom;

        if (vos_dev_ioctl(hUvc, &uvc_iocb) != UVC_OK)
        {
            asm{HALT};
        }

        break;
    } while (1);


    vos_delay_msecs(300);

    vos_wait_semaphore(&sb1_avail);

    status = vos_dev_read(hUvc,buffer1,ISO_TRANSFER_SIZE * ISO_TRANSFER_COUNT,NULL);

    vos_signal_semaphore(&sb1_full);

    do
    {
        vos_wait_semaphore(&sb2_avail);
        // now start the read from the ISO endpoint
        status = vos_dev_read(hUvc,buffer2,ISO_TRANSFER_SIZE * ISO_TRANSFER_COUNT,NULL);
        vos_signal_semaphore(&sb2_full);

        vos_wait_semaphore(&sb1_avail);
        // read from the ISO endpoint
        status = vos_dev_read(hUvc,buffer1,ISO_TRANSFER_SIZE * ISO_TRANSFER_COUNT,NULL);
        vos_signal_semaphore(&sb1_full);

    } while (1);
}



void displayOutput(void)
{

   SEPS525_reset();
   OLED_Init();

   RGB_IF_Init();


    do
    {
        vos_wait_semaphore(&sb1_full);
        process_data_stream(buffer1,BUF_SIZE);
        vos_signal_semaphore(&sb1_avail);

        vos_wait_semaphore(&sb2_full);
        process_data_stream(buffer2,BUF_SIZE);
        vos_signal_semaphore(&sb2_avail);

    }while(1);
}

unsigned char Blue, Green, Red;
unsigned char Y0, Y1, U, V;
unsigned char *inBuf;
unsigned char size;

void YUY2RGBConvert (unsigned char *inputBuffer1 )
{
    inBuf = inputBuffer1;

     while (size)
     {
         //size -= 4;
         asm { DEC8 size $4; };

         //  Y0 = (*inBuf++) - 16;
        asm {
            CPY16    %r0    inBuf
            INC16    inBuf    $1
            CPY8    %r0    (%r0)
            SUB8    %r0    $16
            CPY8    Y0    %r0
            }
           //U = (*inBuf++) - 128;
        asm {
            CPY16    %r0    inBuf
            INC16    inBuf    $1
            CPY8    %r0    (%r0)
            SUB8    %r0    $128
            CPY8    U    %r0
            }
           //Y1 = (*inBuf++) - 16;
        asm {
            CPY16    %r0    inBuf
            INC16    inBuf    $1
            CPY8    %r0    (%r0)
            SUB8    %r0    $16
            CPY8    Y1    %r0
            }
           //V = (*inBuf++) - 128;
        asm {
            CPY16    %r0    inBuf
            INC16    inBuf    $1
            CPY8    %r0    (%r0)
            SUB8    %r0    $128
            CPY8    V    %r0
            }

           //Red = Y0 + V;
        asm {
            ADD8    Red    Y0    V;
            }
        vII_gpio_data_tx_pb_1 = 0x60;
        vII_gpio_data_tx_pa_1 = Red;
        vII_gpio_data_tx_pb_1 = 0x70;

           //Green = Y0 - V - U;
        asm {
            SUB8    Green    Y0    V;
            SUB8    Green    U;
            }
        vII_gpio_data_tx_pb_1 = 0x60;
        vII_gpio_data_tx_pa_1 = Green;
        vII_gpio_data_tx_pb_1 = 0x70;

           //Blue = Y0 + U ;
        asm {
            ADD8    Blue    Y0    U;
            }
        vII_gpio_data_tx_pb_1 = 0x60;
        vII_gpio_data_tx_pa_1 = Blue;
        vII_gpio_data_tx_pb_1 = 0x70;

           //Red = Y1 + V;
        asm {
            ADD8    Red    Y1    V;
            }
        vII_gpio_data_tx_pb_1 = 0x60;
        vII_gpio_data_tx_pa_1 = Red;
        vII_gpio_data_tx_pb_1 = 0x70;

           //Green = Y1 - V - U;
        asm {
            SUB8    Green    Y1    V;
            SUB8    Green    U;
            }
        vII_gpio_data_tx_pb_1 = 0x60;
        vII_gpio_data_tx_pa_1 = Green;
        vII_gpio_data_tx_pb_1 = 0x70;

           //Blue = Y1 + U;
        asm {
            ADD8    Blue    Y1    U;
            }
        vII_gpio_data_tx_pb_1 = 0x60;
        vII_gpio_data_tx_pa_1 = Blue;
        vII_gpio_data_tx_pb_1 = 0x70;

       }
       return;

}



enum {
    SYNCING,
    SYNCED,
    RESYNC,

};

#define WEBCAM_EOF_BIT        0x02
#define WEBCAM_TOG_BIT        0x01
#define WEBCAM_ERR_BIT        0x40

static unsigned char state = SYNCING;
static unsigned short numXfers = 0;
void process_data_stream(unsigned char *buf,unsigned short num_bytes)
{
    unsigned char c;

    while (num_bytes != 0) {

        switch (state) {

        case SYNCING : // syncing with frame header
            if (*buf == 0x0c) {
                // could be a start of a webcam frame
                if ((*(buf+1) & WEBCAM_EOF_BIT) == WEBCAM_EOF_BIT) {
                    // Yup - it's synced to the webcam frames
                    // and goto SYNCED
                    numXfers = 0;
                    state = SYNCED;
                    num_bytes = 0;
                    break;
                }
                // Nope - it's not synced yet
                // consume the data
                num_bytes = 0;
            }
            else {
                asm{HALT};
            }
            break;

        case SYNCED : // synced to webcam frames
            if (*buf == 0x0c) {
                if ((*(buf+1) & WEBCAM_ERR_BIT) == WEBCAM_ERR_BIT) {
                    state = RESYNC;
                    num_bytes = 0;
                    break;
                }
                ++numXfers;
                size = numXfers==1494?60:180;
                YUY2RGBConvert(buf + 12);
                if (numXfers==1494) {
                    numXfers = 0;
                    state = RESYNC;
                    num_bytes = 0;
                    break;
                }

                // consume the payboad
                num_bytes = 0;
            }
            else {
                asm{HALT};
            }
            break;

        case RESYNC : // syncing with frame header
            if (*buf == 0x0c) {
                // could be a start of a webcam frame
                if ((*(buf+1) & WEBCAM_EOF_BIT) == WEBCAM_EOF_BIT) {
                    // Yup - it's synced to the webcam frames
                    // and goto SYNCED
                    numXfers = 0;
                    state = SYNCED;
                    num_bytes = 0;
                    break;
                }
                // Nope - it's not synced yet
                // consume the data
                num_bytes = 0;
            }
            else {
                asm{HALT};
            }
            break;

        default :
            break;

        }
    }
}




