/*
** UVC.c
**
** Copyright  2010 Future Devices International Limited
**
** Company Confidential
**
** C Source file for Vinculum II USB Video Class driver
** Main module
**
** Author:
** Project: Vinculum II
** Module: Vinculum II USB Video Class Driver
** Requires: VOS
** Comments:
**
** History:
**  1  Initial version
**
*/

#include "vos.h"
#include "devman.h"
#include "memmgmt.h"

#include "USB.h"
#include "USBHost.h"

#include "UVC.h"
#include "UVC_defs.h"


static vos_driver_t uvc_cb;


unsigned char uvc_init(unsigned char vos_dev_num)
{
    uvc_context_t *ctx;

    // allocate  context
    ctx = vos_malloc(sizeof(uvc_context_t));
    if (ctx == 0)
    {
        return UVC_ERROR;
    }

    // Set up function pointers to the driver
    uvc_cb.flags = 0;
    uvc_cb.read = uvc_read;
    uvc_cb.write = uvc_write;
    uvc_cb.ioctl = uvc_ioctl;
    uvc_cb.interrupt = (PF_INT) 0;
    uvc_cb.open = uvc_open;
    uvc_cb.close = uvc_close;

    // OK - register with device manager
    vos_dev_init(vos_dev_num, &uvc_cb, ctx);

    return UVC_OK;
}

void uvc_open (uvc_context_t *ctx)
{
    vos_memset(ctx, 0, sizeof(uvc_context_t));
//    ctx->state = WEBCAM_STATE_CLOSED;
}

void uvc_close (uvc_context_t *ctx)
{
    vos_memset(ctx, 0, sizeof(uvc_context_t));
}

unsigned char uvc_detach(uvc_context_t *ctx)
{
    //webcam_closeSession(ctx);
    vos_memset(ctx, 0, sizeof(uvc_context_t));
    return UVC_OK;
}

unsigned char uvc_attach(uvc_context_t *ctx, void *params)
{
    usbhost_ioctl_cb_t hc_iocb;           // USBHost ioctl request block
    usbhost_ioctl_cb_class_t hc_class;
    usbhost_ioctl_cb_dev_info_t devInfo;
    usbhost_device_handle *ifDev;         // device handle
    int ifs, aset;
    unsigned char status;
    unsigned char ret;
    unsigned char num_dev;
    unsigned char i;

    status = ret = UVC_OK;

    ctx->hc = (VOS_HANDLE) params;

    do
    {
        vos_delay_msecs(1000);

        // poll for enumeration to complete
        hc_iocb.ioctl_code = VOS_IOCTL_USBHOST_GET_CONNECT_STATE;
        hc_iocb.get = &i;
        vos_dev_ioctl(ctx->hc, &hc_iocb);


        if (i == PORT_STATE_ENUMERATED)
        {

            num_dev = 0;
            // user ioctl to find device count
            hc_iocb.ioctl_code = VOS_IOCTL_USBHOST_DEVICE_GET_COUNT;
            hc_iocb.set = NULL;
            hc_iocb.get = &num_dev;

            if (vos_dev_ioctl(ctx->hc, &hc_iocb) != USBHOST_OK)
            {
                status = UVC_ERROR;
                break;
            }

            ifDev = NULL;

            for (i = 0; i < num_dev;)
            {
                // user ioctl to find first device
                hc_iocb.ioctl_code = VOS_IOCTL_USBHOST_DEVICE_GET_NEXT_HANDLE;
                hc_iocb.handle.dif = ifDev;
                hc_iocb.get = &ifDev;

                if (vos_dev_ioctl(ctx->hc, &hc_iocb) != USBHOST_OK)
                {
                    ret = UVC_ERROR;
                    break;
                }

                if (ifDev)
                {
                    hc_iocb.ioctl_code = VOS_IOCTL_USBHOST_DEVICE_GET_CLASS_INFO;
                    hc_iocb.get = &hc_class;

                    vos_dev_ioctl(ctx->hc, &hc_iocb);

                    hc_iocb.ioctl_code = VOS_IOCTL_USBHOST_DEVICE_GET_DEV_INFO;
                    hc_iocb.handle.dif = ifDev;
                    hc_iocb.set = NULL;
                    hc_iocb.get = &devInfo;

                    ret = vos_dev_ioctl(ctx->hc, &hc_iocb);

                    if (ret != USBHOST_OK)
                    {
                        ret = UVC_ERROR;
                        break;
                    }


                    ifs = devInfo.interface_number;
                    aset = devInfo.alt;

                    if ((hc_class.dev_class == USB_CLASS_VIDEO) &&
                     (hc_class.dev_subclass == USB_SUBCLASS_VIDEO_VIDEOSTREAMING))
                    {
                        // find interface 1, alternate 1
                        // this is an interface with an isochromous endpoint
                        // it is 192 bytes max packet size
                        if ((ifs == 1) && (aset == 1))
                        {
                            // user ioctl to find iso out endpoint on this device
                            hc_iocb.ioctl_code = VOS_IOCTL_USBHOST_DEVICE_GET_ISO_IN_ENDPOINT_HANDLE;
                            hc_iocb.handle.dif = ifDev;
                            hc_iocb.get = &ctx->epIsoIn;

                            if (vos_dev_ioctl(ctx->hc, &hc_iocb) == USBHOST_OK)
                            {
                                break;
                            }
                        }
                    }
                }
                i++;
            }

            if (ret == UVC_ERROR)
            {
                status = UVC_ERROR;
                break;
            }

            if (ifDev == NULL)
            {
                status = UVC_ERROR;
                break;
            }

            // user ioctl to find interrupt endpoint on this device
            hc_iocb.ioctl_code = VOS_IOCTL_USBHOST_DEVICE_GET_CONTROL_ENDPOINT_HANDLE;
            hc_iocb.handle.dif = ifDev;
            hc_iocb.get = &ctx->epCtrl;

            if (vos_dev_ioctl(ctx->hc, &hc_iocb) != USBHOST_OK)
            {
                status = UVC_ERROR;
                break;
            }


            // set this interface to be enabled
            hc_iocb.ioctl_code = VOS_IOCTL_USBHOST_SET_INTERFACE;
            hc_iocb.handle.dif = ifDev;

            if (vos_dev_ioctl(ctx->hc, &hc_iocb) != USBHOST_OK)
            {
                status = UVC_ERROR;
                break;
            }


            // user ioctl to find interrupt endpoint on this device
            hc_iocb.ioctl_code = VOS_IOCTL_USBHOST_DEVICE_GET_ENDPOINT_INFO;
            hc_iocb.handle.ep = ctx->epIsoIn;
            hc_iocb.get = &ctx->epInfo;

            if (vos_dev_ioctl(ctx->hc, &hc_iocb) != USBHOST_OK)
            {
                status = UVC_ERROR;
                break;
            }

            break;

        }
    } while (1);

    return status;
}

unsigned char host_device_setup_transfer(uvc_context_t *ctx,usb_deviceRequest_t *req_desc,void *out_data)
{
   // USBHost ioctl request block
    usbhost_ioctl_cb_t iocb;

    iocb.ioctl_code = VOS_IOCTL_USBHOST_DEVICE_SETUP_TRANSFER;
    iocb.handle.ep = ctx->epCtrl;
    iocb.set = req_desc;
    iocb.get = out_data;

    if (vos_dev_ioctl(ctx->hc,&iocb) != USBHOST_OK)
        return UVC_ERROR;

    return UVC_OK;
}

unsigned char uvc_class_request(uvc_context_t *ctx,void *in_data,void *out_data)
{
    usb_deviceRequest_t *req = (usb_deviceRequest_t *) in_data;

    if (req->bRequest == SET_CUR) {
        req->bmRequestType = USB_BMREQUESTTYPE_HOST_TO_DEV
                    | USB_BMREQUESTTYPE_CLASS
                    | USB_BMREQUESTTYPE_INTERFACE;
    }
    else {
        req->bmRequestType = USB_BMREQUESTTYPE_DEV_TO_HOST
                        | USB_BMREQUESTTYPE_CLASS
                        | USB_BMREQUESTTYPE_INTERFACE;
    }

    return host_device_setup_transfer(ctx,req,out_data);
}

// UVC read function
unsigned char uvc_read (char *buffer,
    unsigned short num_to_read,
    unsigned short *num_read,
    uvc_context_t *ctx)
{
    usbhost_ioctl_cb_t hc_iocb;
    usbhost_xfer_iso_t xfer;
    vos_semaphore_t semDum;
    unsigned char status;
    unsigned short frame;
    unsigned char i=0;
    unsigned short xfer_size = ctx->epInfo.max_size;

    vos_init_semaphore(&semDum, 0);
    // form isochronous transfer descriptor
    xfer.cond_code = USBHOST_CC_NOTACCESSED;
    xfer.flags = 0;
    xfer.s = &semDum;
    xfer.ep = ctx->epIsoIn;
    xfer.buf = buffer;
    xfer.len = num_to_read;

    xfer.count = (unsigned char) (num_to_read/xfer_size);
    while (num_to_read) {
        xfer.len_psw[i].size = (num_to_read>xfer_size) ? xfer_size : num_to_read;
        num_to_read -= xfer_size;
        i++;
    }

    // user ioctl to find frame number (as close as possible to use)
    hc_iocb.ioctl_code = VOS_IOCTL_USBHOST_HW_GET_FRAME_NUMBER;
    hc_iocb.get = &frame;

    if (vos_dev_ioctl(ctx->hc, &hc_iocb) != USBHOST_OK)
    {
        return UVC_ERROR;
    }

    xfer.frame = frame + 1;

    // now start the read from the ISO endpoint
    status = vos_dev_read(ctx->hc, (unsigned char *)&xfer, sizeof(usbhost_xfer_t), NULL);
    if (status != USBHOST_OK)
    {
        return UVC_ERROR;
    }

    if (xfer.cond_code != USBHOST_CC_NOERROR)
    {
        return UVC_ERROR;
    }

    return UVC_OK;
}

// UVC write function
unsigned char uvc_write (char *buffer,
    unsigned short num_to_write,
    unsigned short *num_written,
    uvc_context_t *ctx)
{
    unsigned char status = UVC_OK;
    return status;
}


// UVC IOCTL function
unsigned char uvc_ioctl(uvc_ioctl_cb_t *cb, uvc_context_t *ctx)
{
    unsigned char status = UVC_INVALID_PARAMETER;

    switch (cb->ioctl_code) {

    case VOS_IOCTL_UVC_ATTACH:
        status = uvc_attach(ctx,cb->set);
        break;

    case VOS_IOCTL_UVC_DETACH:
        status = uvc_detach(ctx);
        break;

    case VOS_IOCTL_UVC_CLASS_REQUEST:
        status = uvc_class_request(ctx,cb->set,cb->get);
        break;

    default:
        break;

    }

    return status;
}



