/*

Copyright (c) Future Technology Devices International 2014

THIS SOFTWARE IS PROVIDED BY FUTURE TECHNOLOGY DEVICES INTERNATIONAL LIMITED "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
FUTURE TECHNOLOGY DEVICES INTERNATIONAL LIMITED BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
OF SUBSTITUTE GOODS OR SERVICES LOSS OF USE, DATA, OR PROFITS OR BUSINESS INTERRUPTION)
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

FTDI DRIVERS MAY BE USED ONLY IN CONJUNCTION WITH PRODUCTS BASED ON FTDI PARTS.

FTDI DRIVERS MAY BE DISTRIBUTED IN ANY FORM AS LONG AS LICENSE INFORMATION IS NOT MODIFIED.

IF A CUSTOM VENDOR ID AND/OR PRODUCT ID OR DESCRIPTION STRING ARE USED, IT IS THE
RESPONSIBILITY OF THE PRODUCT MANUFACTURER TO MAINTAIN ANY CHANGES AND SUBSEQUENT WHQL
RE-CERTIFICATION AS A RESULT OF MAKING THESE CHANGES.

Abstract:

This file contains is functions for all UI fields.

Author : FTDI

Revision History: 
0.1 - date 2013.04.24 - initial version
0.2 - date 2013.08.19 - few minor edits

*/

#ifndef FT800EMU_EMULATOR_H
#define FT800EMU_EMULATOR_H


// System includes
#include <string>

// Project includes (include standard stuff for user)
#include "FT_DataTypes.h"

namespace FT800EMU {

enum EmulatorFlags
{
	// enables the keyboard to be used as input
	EmulatorEnableKeyboard = 0x01,
	// enables audio
	EmulatorEnableAudio = 0x02,
	// enables coprocessor
	EmulatorEnableCoprocessor = 0x04,
	// enables mouse as touch
	EmulatorEnableMouse = 0x08, 
	// enable debug shortkeys
	EmulatorEnableDebugShortkeys = 0x10, 
	// enable graphics processor multithreading
	EmulatorEnableGraphicsMultithread = 0x20, 
	// enable dynamic graphics quality degrading by interlacing
	EmulatorEnableDynamicDegrade = 0x40, 
	// enable usage of REG_ROTATE
	EmulatorEnableRegRotate = 0x80, 
	// enable emulating REG_PWM_DUTY by fading the rendered display to black
	EmulatorEnableRegPwmDutyEmulation = 0x100, 
	// enable killing the loop thread on exit
	// EmulatorEnableKillLoopThread = 0x200, 
};

struct EmulatorParameters
{
public:
	EmulatorParameters() : 
		Setup(0),
		Loop(0), 
		Flags(0),
		Keyboard(0), 
		MousePressure(0), 
		ExternalFrequency(0), 
		ReduceGraphicsThreads(0),
		Graphics(0)
	{ }

	// Microcontroller function called before loop.
	void (*Setup)();
	// Microcontroller continuous loop.
	void (*Loop)();
	// See EmulatorFlags.
	int Flags;
	
	// Called after keyboard update.
	// Supplied function can use Keyboard.isKeyDown(FT800EMU_KEY_F3).
	void (*Keyboard)();
	// The default mouse pressure, default 0 (maximum).
	// See REG_TOUCH_RZTRESH, etc.
	uint32_t MousePressure;
	// External frequency. See CLK, etc.
	uint32_t ExternalFrequency;
	
	// Reduce graphics processor threads by specified number, default 0
	// Necessary when doing very heavy work on the MCU or Coprocessor
	// TODO: Maybe possible to automate this based on thread info
	uint32_t ReduceGraphicsThreads;

	// Replaces the default builtin ROM with a custom ROM from a file.
	std::string RomFilePath;
	// Replaces the builtin coprocessor ROM.
	std::string CoprocessorRomFilePath;
	
	// Graphics driverless mode
	// Setting this callback means no window will be created, and all
	// rendered graphics will be automatically sent to this function.
	// For enabling touch functionality, the functions 
	// Memory.setTouchScreenXY and Memory.resetTouchScreenXY must be
	// called manually from the host application.
	// Builtin keyboard functionality is not supported and must be
	// implemented manually when using this mode.
	// The output parameter is false when the display is turned off.
	// The contents of the buffer pointer are undefined after this
	// function returns.
	// Return false when the application must exit.
	bool (*Graphics)(bool output, const argb8888 *buffer, uint32_t hsize, uint32_t vsize);
};

class EmulatorClass
{
public:
	EmulatorClass() { }

	static void run(const EmulatorParameters &params);
	static void stop();

private:
	EmulatorClass(const EmulatorClass &);
	EmulatorClass &operator=(const EmulatorClass &);

}; /* class EmulatorClass */

extern EmulatorClass Emulator;

} /* namespace FT800EMU */

#endif /* #ifndef FT800EMU_EMULATOR_H */

/* end of file */
